<?php
declare(strict_types=1);

require __DIR__ . '/db.php';
$cfg = require __DIR__ . '/config.php';

header('Content-Type: application/json; charset=utf-8');

function json_ok(array $data = []): void {
  echo json_encode(['ok' => true] + $data, JSON_UNESCAPED_UNICODE);
  exit;
}
function json_err(string $msg, int $code = 400): void {
  http_response_code($code);
  echo json_encode(['ok' => false, 'error' => $msg], JSON_UNESCAPED_UNICODE);
  exit;
}

$action = (string)($_GET['action'] ?? '');

try {
  $pdo = db();

  if ($action === 'tracks') {
    $q = trim((string)($_GET['q'] ?? ''));
    if ($q !== '') {
      $stmt = $pdo->prepare("SELECT * FROM tracks WHERE title LIKE :q OR artist LIKE :q OR album LIKE :q ORDER BY id DESC LIMIT 500");
      $stmt->execute([':q' => '%' . $q . '%']);
    } else {
      $stmt = $pdo->query("SELECT * FROM tracks ORDER BY id DESC LIMIT 500");
    }
    $tracks = $stmt->fetchAll();
    $tracks = array_map(function($t) use ($cfg) {
      $t['url'] = $cfg['base_url_uploads'] . '/' . $t['filename'];
      return $t;
    }, $tracks);
    json_ok(['tracks' => $tracks]);
  }

  if ($action === 'delete_track') {
    $id = (int)($_POST['id'] ?? 0);
    if ($id <= 0) json_err('شناسه ترک نامعتبر است.');
    $stmt = $pdo->prepare("SELECT filename FROM tracks WHERE id = :id");
    $stmt->execute([':id' => $id]);
    $row = $stmt->fetch();
    if (!$row) json_err('ترک پیدا نشد.', 404);

    $pdo->beginTransaction();
    $pdo->prepare("DELETE FROM tracks WHERE id = :id")->execute([':id' => $id]);
    $pdo->commit();

    $file = rtrim($cfg['upload_dir'], '/\\') . DIRECTORY_SEPARATOR . $row['filename'];
    if (is_file($file)) @unlink($file);

    json_ok();
  }

  if ($action === 'playlists') {
    $pls = $pdo->query("SELECT * FROM playlists ORDER BY id DESC")->fetchAll();
    json_ok(['playlists' => $pls]);
  }

  if ($action === 'create_playlist') {
    $name = trim((string)($_POST['name'] ?? ''));
    if ($name === '') json_err('نام پلی‌لیست خالی است.');
    $stmt = $pdo->prepare("INSERT INTO playlists(name) VALUES(:name)");
    $stmt->execute([':name' => $name]);
    json_ok(['id' => (int)$pdo->lastInsertId()]);
  }

  if ($action === 'rename_playlist') {
    $id = (int)($_POST['id'] ?? 0);
    $name = trim((string)($_POST['name'] ?? ''));
    if ($id <= 0 || $name === '') json_err('ورودی نامعتبر است.');
    $stmt = $pdo->prepare("UPDATE playlists SET name = :name WHERE id = :id");
    $stmt->execute([':name' => $name, ':id' => $id]);
    json_ok();
  }

  if ($action === 'delete_playlist') {
    $id = (int)($_POST['id'] ?? 0);
    if ($id <= 0) json_err('شناسه نامعتبر است.');
    $stmt = $pdo->prepare("DELETE FROM playlists WHERE id = :id");
    $stmt->execute([':id' => $id]);
    json_ok();
  }

  if ($action === 'playlist_items') {
    $playlistId = (int)($_GET['playlist_id'] ?? 0);
    if ($playlistId <= 0) json_err('شناسه پلی‌لیست نامعتبر است.');
    $stmt = $pdo->prepare("
      SELECT t.*, pi.position
      FROM playlist_items pi
      JOIN tracks t ON t.id = pi.track_id
      WHERE pi.playlist_id = :pid
      ORDER BY pi.position ASC, t.id ASC
    ");
    $stmt->execute([':pid' => $playlistId]);
    $items = $stmt->fetchAll();
    $items = array_map(function($t) use ($cfg) {
      $t['url'] = $cfg['base_url_uploads'] . '/' . $t['filename'];
      return $t;
    }, $items);
    json_ok(['items' => $items]);
  }

  if ($action === 'add_to_playlist') {
    $playlistId = (int)($_POST['playlist_id'] ?? 0);
    $trackId = (int)($_POST['track_id'] ?? 0);
    if ($playlistId <= 0 || $trackId <= 0) json_err('ورودی نامعتبر است.');

    // next position
    $stmt = $pdo->prepare("SELECT COALESCE(MAX(position), -1) AS maxpos FROM playlist_items WHERE playlist_id = :pid");
    $stmt->execute([':pid' => $playlistId]);
    $maxpos = (int)($stmt->fetch()['maxpos'] ?? -1);
    $pos = $maxpos + 1;

    $stmt = $pdo->prepare("
      INSERT OR IGNORE INTO playlist_items(playlist_id, track_id, position)
      VALUES(:pid, :tid, :pos)
    ");
    $stmt->execute([':pid' => $playlistId, ':tid' => $trackId, ':pos' => $pos]);
    json_ok();
  }

  if ($action === 'remove_from_playlist') {
    $playlistId = (int)($_POST['playlist_id'] ?? 0);
    $trackId = (int)($_POST['track_id'] ?? 0);
    if ($playlistId <= 0 || $trackId <= 0) json_err('ورودی نامعتبر است.');
    $stmt = $pdo->prepare("DELETE FROM playlist_items WHERE playlist_id = :pid AND track_id = :tid");
    $stmt->execute([':pid' => $playlistId, ':tid' => $trackId]);
    json_ok();
  }

  json_err('اکشن نامعتبر است.', 404);

} catch (Throwable $e) {
  json_err('خطا: ' . $e->getMessage(), 500);
}
