const $ = (sel) => document.querySelector(sel);

const audio = $("#audio");
const nowTitle = $("#nowTitle");
const nowArtist = $("#nowArtist");
const trackListEl = $("#trackList");
const playlistListEl = $("#playlistList");
const playlistItemsEl = $("#playlistItems");
const plTitleEl = $("#plTitle");

const btnPlay = $("#btnPlay");
const btnPrev = $("#btnPrev");
const btnNext = $("#btnNext");
const seek = $("#seek");
const tCur = $("#tCur");
const tDur = $("#tDur");
const vol = $("#vol");
const search = $("#search");

const btnRefresh = $("#btnRefresh");
const btnUploadOpen = $("#btnUploadOpen");
const btnUploadClose = $("#btnUploadClose");
const uploadBackdrop = $("#uploadBackdrop");
const uploadSheet = $("#uploadSheet");
const uploadForm = $("#uploadForm");
const uploadProgress = $("#uploadProgress");
const uploadBar = $("#uploadBar");

const btnNewPlaylist = $("#btnNewPlaylist");
const toast = $("#toast");

let tracks = [];
let currentIndex = -1;
let currentPlaylistId = null;
let playlistItems = [];

function toastMsg(msg) {
  toast.textContent = msg;
  toast.hidden = false;
  setTimeout(() => { toast.hidden = true; }, 2500);
}

function fmtTime(sec) {
  if (!isFinite(sec)) return "0:00";
  sec = Math.max(0, Math.floor(sec));
  const m = Math.floor(sec / 60);
  const s = sec % 60;
  return `${m}:${String(s).padStart(2, "0")}`;
}

async function apiGet(action, params = {}) {
  const url = new URL("api.php", location.href);
  url.searchParams.set("action", action);
  Object.entries(params).forEach(([k, v]) => url.searchParams.set(k, v));
  const res = await fetch(url.toString());
  const data = await res.json();
  if (!data.ok) throw new Error(data.error || "API error");
  return data;
}
async function apiPost(action, body = {}) {
  const form = new FormData();
  Object.entries(body).forEach(([k, v]) => form.append(k, v));
  const res = await fetch(`api.php?action=${encodeURIComponent(action)}`, { method: "POST", body: form });
  const data = await res.json();
  if (!data.ok) throw new Error(data.error || "API error");
  return data;
}

function getActiveQueue() {
  // if a playlist is selected, play from playlistItems, otherwise from all tracks
  if (currentPlaylistId && playlistItems.length) return playlistItems;
  return tracks;
}

function setNow(track) {
  nowTitle.textContent = track?.title || "—";
  nowArtist.textContent = track?.artist || track?.album || "—";
}

function playIndex(idx) {
  const queue = getActiveQueue();
  if (!queue.length) return;

  idx = (idx + queue.length) % queue.length;
  currentIndex = idx;

  const tr = queue[idx];
  setNow(tr);
  audio.src = tr.url;
  audio.play().catch(() => {});
  renderTrackList();
  renderPlaylistItems();
}

function togglePlay() {
  if (!audio.src) {
    // start first track if any
    const queue = getActiveQueue();
    if (queue.length) return playIndex(0);
    return;
  }
  if (audio.paused) audio.play().catch(() => {});
  else audio.pause();
}

function updatePlayBtn() {
  btnPlay.textContent = audio.paused ? "▶︎" : "❚❚";
}

function isCurrentTrackId(id) {
  const queue = getActiveQueue();
  if (currentIndex < 0 || currentIndex >= queue.length) return false;
  return queue[currentIndex]?.id === id;
}

function renderTrackList() {
  trackListEl.innerHTML = "";
  const q = search.value.trim().toLowerCase();
  const filtered = q
    ? tracks.filter(t =>
        (t.title || "").toLowerCase().includes(q) ||
        (t.artist || "").toLowerCase().includes(q) ||
        (t.album || "").toLowerCase().includes(q)
      )
    : tracks;

  if (!filtered.length) {
    trackListEl.innerHTML = `<div class="item"><div class="left"><div class="title">چیزی پیدا نشد</div><div class="sub">آپلود کن یا جستجو را تغییر بده</div></div></div>`;
    return;
  }

  filtered.forEach((t) => {
    const item = document.createElement("div");
    item.className = "item";
    const current = isCurrentTrackId(t.id);
    item.innerHTML = `
      <div class="left">
        <div class="title">${escapeHtml(t.title)}</div>
        <div class="sub">${escapeHtml(t.artist || t.album || "—")}</div>
      </div>
      <div class="right">
        ${current ? `<span class="badge">درحال پخش</span>` : ``}
        <button class="mini-btn" data-act="play">پخش</button>
        <button class="mini-btn" data-act="add">+ پلی‌لیست</button>
        <button class="mini-btn danger" data-act="del">حذف</button>
      </div>
    `;

    item.querySelector('[data-act="play"]').onclick = () => {
      // find this track in active queue if playlist mode; else in tracks
      if (currentPlaylistId && playlistItems.length) {
        const idx = playlistItems.findIndex(x => x.id === t.id);
        if (idx >= 0) playIndex(idx);
        else {
          // fallback: play from library
          currentPlaylistId = null;
          plTitleEl.textContent = "—";
          playIndex(tracks.findIndex(x => x.id === t.id));
        }
      } else {
        playIndex(tracks.findIndex(x => x.id === t.id));
      }
    };

    item.querySelector('[data-act="add"]').onclick = async () => {
      if (!currentPlaylistId) {
        toastMsg("اول یک پلی‌لیست انتخاب کن.");
        return;
      }
      try {
        await apiPost("add_to_playlist", { playlist_id: currentPlaylistId, track_id: t.id });
        toastMsg("به پلی‌لیست اضافه شد.");
        await loadPlaylistItems(currentPlaylistId);
      } catch (e) {
        toastMsg(e.message);
      }
    };

    item.querySelector('[data-act="del"]').onclick = async () => {
      if (!confirm("این ترک حذف شود؟")) return;
      try {
        await apiPost("delete_track", { id: t.id });
        toastMsg("حذف شد.");
        await loadAll();
      } catch (e) {
        toastMsg(e.message);
      }
    };

    trackListEl.appendChild(item);
  });
}

function renderPlaylists(playlists) {
  playlistListEl.innerHTML = "";
  if (!playlists.length) {
    playlistListEl.innerHTML = `<div class="pl"><div class="name">پلی‌لیستی نداری</div><div class="tools"><span class="badge">بساز</span></div></div>`;
    return;
  }

  playlists.forEach((pl) => {
    const el = document.createElement("div");
    el.className = "pl" + (pl.id === currentPlaylistId ? " active" : "");
    el.innerHTML = `
      <div class="name">${escapeHtml(pl.name)}</div>
      <div class="tools">
        <button class="mini-btn" data-act="open">باز</button>
        <button class="mini-btn" data-act="ren">نام</button>
        <button class="mini-btn danger" data-act="del">حذف</button>
      </div>
    `;

    el.querySelector('[data-act="open"]').onclick = async (ev) => {
      ev.stopPropagation();
      currentPlaylistId = pl.id;
      plTitleEl.textContent = pl.name;
      await loadPlaylistItems(pl.id);
      renderTrackList();
    };

    el.onclick = async () => {
      currentPlaylistId = pl.id;
      plTitleEl.textContent = pl.name;
      await loadPlaylistItems(pl.id);
      renderPlaylists(playlists);
      renderTrackList();
    };

    el.querySelector('[data-act="ren"]').onclick = async (ev) => {
      ev.stopPropagation();
      const name = prompt("نام جدید پلی‌لیست:", pl.name);
      if (!name) return;
      try {
        await apiPost("rename_playlist", { id: pl.id, name });
        toastMsg("ذخیره شد.");
        await loadPlaylists();
      } catch (e) {
        toastMsg(e.message);
      }
    };

    el.querySelector('[data-act="del"]').onclick = async (ev) => {
      ev.stopPropagation();
      if (!confirm("این پلی‌لیست حذف شود؟")) return;
      try {
        await apiPost("delete_playlist", { id: pl.id });
        toastMsg("حذف شد.");
        if (currentPlaylistId === pl.id) {
          currentPlaylistId = null;
          playlistItems = [];
          plTitleEl.textContent = "—";
          playlistItemsEl.innerHTML = "";
        }
        await loadPlaylists();
      } catch (e) {
        toastMsg(e.message);
      }
    };

    playlistListEl.appendChild(el);
  });
}

function renderPlaylistItems() {
  playlistItemsEl.innerHTML = "";
  if (!currentPlaylistId) {
    playlistItemsEl.innerHTML = `<div class="item"><div class="left"><div class="title">پلی‌لیست انتخاب نشده</div><div class="sub">از سمت راست یکی را انتخاب کن</div></div></div>`;
    return;
  }
  if (!playlistItems.length) {
    playlistItemsEl.innerHTML = `<div class="item"><div class="left"><div class="title">خالی است</div><div class="sub">از کتابخانه «+ پلی‌لیست» بزن</div></div></div>`;
    return;
  }

  playlistItems.forEach((t, idx) => {
    const item = document.createElement("div");
    item.className = "item";
    const current = isCurrentTrackId(t.id);
    item.innerHTML = `
      <div class="left">
        <div class="title">${escapeHtml(t.title)}</div>
        <div class="sub">${escapeHtml(t.artist || t.album || "—")}</div>
      </div>
      <div class="right">
        ${current ? `<span class="badge">درحال پخش</span>` : ``}
        <button class="mini-btn" data-act="play">پخش</button>
        <button class="mini-btn danger" data-act="rm">حذف</button>
      </div>
    `;

    item.querySelector('[data-act="play"]').onclick = () => playIndex(idx);

    item.querySelector('[data-act="rm"]').onclick = async () => {
      try {
        await apiPost("remove_from_playlist", { playlist_id: currentPlaylistId, track_id: t.id });
        toastMsg("از پلی‌لیست حذف شد.");
        await loadPlaylistItems(currentPlaylistId);
      } catch (e) {
        toastMsg(e.message);
      }
    };

    playlistItemsEl.appendChild(item);
  });
}

async function loadTracks() {
  const data = await apiGet("tracks");
  tracks = data.tracks || [];
  renderTrackList();
}

async function loadPlaylists() {
  const data = await apiGet("playlists");
  renderPlaylists(data.playlists || []);
}

async function loadPlaylistItems(pid) {
  const data = await apiGet("playlist_items", { playlist_id: pid });
  playlistItems = data.items || [];
  renderPlaylistItems();
}

async function loadAll() {
  await Promise.all([loadTracks(), loadPlaylists()]);
  if (currentPlaylistId) {
    try { await loadPlaylistItems(currentPlaylistId); } catch {}
  } else {
    renderPlaylistItems();
  }
}

function openUpload() {
  uploadBackdrop.hidden = false;
  uploadSheet.hidden = false;
}
function closeUpload() {
  uploadBackdrop.hidden = true;
  uploadSheet.hidden = true;
  uploadProgress.hidden = true;
  uploadBar.style.width = "0%";
  uploadForm.reset();
}

btnUploadOpen.onclick = openUpload;
btnUploadClose.onclick = closeUpload;
uploadBackdrop.onclick = closeUpload;

btnRefresh.onclick = () => loadAll().catch(e => toastMsg(e.message));

btnNewPlaylist.onclick = async () => {
  const name = prompt("نام پلی‌لیست:");
  if (!name) return;
  try {
    await apiPost("create_playlist", { name });
    toastMsg("پلی‌لیست ساخته شد.");
    await loadPlaylists();
  } catch (e) {
    toastMsg(e.message);
  }
};

search.oninput = () => renderTrackList();

btnPlay.onclick = togglePlay;
btnPrev.onclick = () => playIndex(currentIndex - 1);
btnNext.onclick = () => playIndex(currentIndex + 1);

audio.addEventListener("play", updatePlayBtn);
audio.addEventListener("pause", updatePlayBtn);
audio.addEventListener("ended", () => playIndex(currentIndex + 1));

audio.addEventListener("loadedmetadata", () => {
  tDur.textContent = fmtTime(audio.duration);
});

audio.addEventListener("timeupdate", () => {
  tCur.textContent = fmtTime(audio.currentTime);
  if (isFinite(audio.duration) && audio.duration > 0) {
    seek.value = String(Math.floor((audio.currentTime / audio.duration) * 100));
  }
});

seek.addEventListener("input", () => {
  if (!isFinite(audio.duration) || audio.duration <= 0) return;
  const pct = Number(seek.value) / 100;
  audio.currentTime = pct * audio.duration;
});

vol.addEventListener("input", () => {
  audio.volume = Number(vol.value);
});
audio.volume = Number(vol.value);

uploadForm.addEventListener("submit", async (ev) => {
  ev.preventDefault();
  const file = $("#file").files[0];
  if (!file) return;

  uploadProgress.hidden = false;
  uploadBar.style.width = "0%";

  const formData = new FormData();
  formData.append("file", file);

  try {
    // Use XHR for progress
    const res = await xhrUpload("upload.php", formData, (pct) => {
      uploadBar.style.width = `${pct}%`;
    });

    if (!res.ok) throw new Error(res.error || "آپلود ناموفق");
    toastMsg("آپلود شد.");
    closeUpload();
    await loadTracks();
  } catch (e) {
    toastMsg(e.message);
  }
});

function xhrUpload(url, formData, onProgress) {
  return new Promise((resolve, reject) => {
    const xhr = new XMLHttpRequest();
    xhr.open("POST", url, true);
    xhr.responseType = "json";
    xhr.upload.onprogress = (e) => {
      if (e.lengthComputable) {
        const pct = Math.round((e.loaded / e.total) * 100);
        onProgress(pct);
      }
    };
    xhr.onload = () => resolve(xhr.response || { ok: false, error: "پاسخ نامعتبر" });
    xhr.onerror = () => reject(new Error("خطای شبکه"));
    xhr.send(formData);
  });
}

function escapeHtml(str) {
  return String(str ?? "")
    .replaceAll("&", "&amp;")
    .replaceAll("<", "&lt;")
    .replaceAll(">", "&gt;")
    .replaceAll('"', "&quot;")
    .replaceAll("'", "&#039;");
}

// Boot
loadAll().catch(e => toastMsg(e.message));
