<?php
declare(strict_types=1);

require __DIR__ . '/db.php';
$cfg = require __DIR__ . '/config.php';

header('Content-Type: application/json; charset=utf-8');

function json_ok(array $data = []): void {
  echo json_encode(['ok' => true] + $data, JSON_UNESCAPED_UNICODE);
  exit;
}
function json_err(string $msg, int $code = 400): void {
  http_response_code($code);
  echo json_encode(['ok' => false, 'error' => $msg], JSON_UNESCAPED_UNICODE);
  exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  json_err('فقط POST مجاز است.', 405);
}

try {
  if (!isset($_FILES['file'])) json_err('فایلی ارسال نشده.');

  $file = $_FILES['file'];
  if (!is_array($file) || $file['error'] !== UPLOAD_ERR_OK) {
    json_err('آپلود ناموفق است.');
  }

  $maxBytes = (int)$cfg['max_upload_mb'] * 1024 * 1024;
  if ((int)$file['size'] > $maxBytes) {
    json_err('حجم فایل بیشتر از حد مجاز است.');
  }

  $origName = (string)$file['name'];
  $ext = strtolower(pathinfo($origName, PATHINFO_EXTENSION));
  if (!in_array($ext, $cfg['allowed_ext'], true)) {
    json_err('فرمت فایل مجاز نیست.');
  }

  $finfo = new finfo(FILEINFO_MIME_TYPE);
  $mime = (string)$finfo->file($file['tmp_name']);

  if (!in_array($mime, $cfg['allowed_mime'], true)) {
    // allow common mis-detection for mp3/m4a by extension
    if (!in_array($ext, ['mp3','m4a','wav','ogg'], true)) {
      json_err('نوع فایل (MIME) مجاز نیست: ' . $mime);
    }
  }

  if (!is_dir($cfg['upload_dir'])) {
    if (!mkdir($cfg['upload_dir'], 0755, true)) {
      json_err('ساخت پوشه آپلود ممکن نیست.', 500);
    }
  }

  // Make a safe, unique filename
  $safeBase = preg_replace('/[^a-zA-Z0-9_-]+/', '_', pathinfo($origName, PATHINFO_FILENAME));
  $safeBase = trim($safeBase, '_');
  if ($safeBase === '') $safeBase = 'track';
  $unique = bin2hex(random_bytes(8));
  $newName = $safeBase . '_' . $unique . '.' . $ext;

  $dest = rtrim($cfg['upload_dir'], '/\\') . DIRECTORY_SEPARATOR . $newName;
  if (!move_uploaded_file($file['tmp_name'], $dest)) {
    json_err('ذخیره فایل روی سرور ناموفق بود.', 500);
  }

  // Basic metadata from filename: "Artist - Title"
  $title = $safeBase;
  $artist = null;
  $album = null;

  $rawBase = pathinfo($origName, PATHINFO_FILENAME);
  if (strpos($rawBase, '-') !== false) {
    $parts = array_map('trim', explode('-', $rawBase, 2));
    if (count($parts) === 2 && $parts[0] !== '' && $parts[1] !== '') {
      $artist = $parts[0];
      $title = $parts[1];
    } else {
      $title = $rawBase;
    }
  } else {
    $title = $rawBase;
  }

  $pdo = db();
  $stmt = $pdo->prepare("
    INSERT INTO tracks(title, artist, album, filename, mime, size_bytes)
    VALUES(:title, :artist, :album, :filename, :mime, :size_bytes)
  ");
  $stmt->execute([
    ':title' => $title,
    ':artist' => $artist,
    ':album' => $album,
    ':filename' => $newName,
    ':mime' => $mime,
    ':size_bytes' => (int)$file['size'],
  ]);

  $id = (int)$pdo->lastInsertId();
  json_ok([
    'id' => $id,
    'url' => $cfg['base_url_uploads'] . '/' . $newName,
    'title' => $title,
    'artist' => $artist,
  ]);

} catch (Throwable $e) {
  json_err('خطا: ' . $e->getMessage(), 500);
}
